import { useState } from "react";
import { Button } from "@/components/ui/button";
import { Dialog, DialogContent, DialogHeader, DialogTitle, DialogTrigger } from "@/components/ui/dialog";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { useToast } from "@/components/ui/use-toast";
import { Upload, Download, FileText } from "lucide-react";
import * as XLSX from "xlsx";

interface ImportDialogProps {
  title: string;
  templateFields: string[];
  onImport: (data: any[]) => Promise<void>;
  isLoading: boolean;
}

export default function ImportDialog({ title, templateFields, onImport, isLoading }: ImportDialogProps) {
  const [isOpen, setIsOpen] = useState(false);
  const [file, setFile] = useState<File | null>(null);
  const { toast } = useToast();

  const downloadTemplate = () => {
    try {
      const wb = XLSX.utils.book_new();
      const ws = XLSX.utils.aoa_to_sheet([templateFields]);
      
      // Set column widths for better readability
      ws['!cols'] = templateFields.map(() => ({ width: 20 }));
      
      XLSX.utils.book_append_sheet(wb, ws, "Template");
      XLSX.writeFile(wb, `template_${title.toLowerCase().replace(/\s+/g, '_')}.xlsx`);
      
      toast({
        title: "Template Downloaded",
        description: "Template berhasil diunduh. Silakan isi data dan upload kembali.",
      });
    } catch (error) {
      console.error("Download template error:", error);
      toast({
        title: "Error",
        description: "Gagal mengunduh template",
        variant: "destructive",
      });
    }
  };

  const handleFileChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    const selectedFile = e.target.files?.[0];
    if (selectedFile) {
      // Reset previous file
      setFile(null);
      
      // Check file extension
      const fileName = selectedFile.name.toLowerCase();
      const validExtensions = ['.xlsx', '.xls', '.csv'];
      const hasValidExtension = validExtensions.some(ext => fileName.endsWith(ext));
      
      if (!hasValidExtension) {
        toast({
          title: "Error",
          description: "Silakan pilih file Excel (.xlsx, .xls) atau CSV (.csv)",
          variant: "destructive",
        });
        e.target.value = ''; // Reset input
        return;
      }
      
      // Check file size (max 5MB)
      if (selectedFile.size > 5 * 1024 * 1024) {
        toast({
          title: "Error",
          description: "Ukuran file terlalu besar. Maksimal 5MB.",
          variant: "destructive",
        });
        e.target.value = ''; // Reset input
        return;
      }
      
      setFile(selectedFile);
      console.log("File selected:", selectedFile.name, selectedFile.type, selectedFile.size);
    }
  };

  const handleImport = async () => {
    if (!file) {
      toast({
        title: "Error",
        description: "Silakan pilih file untuk diimport",
        variant: "destructive",
      });
      return;
    }

    try {
      console.log("Starting import process for file:", file.name);
      
      const data = await file.arrayBuffer();
      console.log("File read successfully, size:", data.byteLength);
      
      const workbook = XLSX.read(data, { 
        type: 'array',
        cellDates: true,
        cellNF: false,
        cellText: false
      });
      
      console.log("Workbook parsed, sheets:", workbook.SheetNames);
      
      if (!workbook.SheetNames.length) {
        throw new Error("No worksheets found in file");
      }
      
      const worksheet = workbook.Sheets[workbook.SheetNames[0]];
      if (!worksheet) {
        throw new Error("Cannot read worksheet");
      }
      
      // Convert to JSON with header row as keys
      const jsonData = XLSX.utils.sheet_to_json(worksheet, { 
        header: 1,
        defval: "",
        blankrows: false
      });

      console.log("Raw JSON data:", jsonData);

      if (jsonData.length === 0) {
        toast({
          title: "Error",
          description: "File kosong atau tidak valid",
          variant: "destructive",
        });
        return;
      }

      if (jsonData.length < 2) {
        toast({
          title: "Error",
          description: "File harus memiliki header dan minimal 1 baris data",
          variant: "destructive",
        });
        return;
      }

      // Get headers and data rows
      const headers = jsonData[0] as string[];
      const dataRows = jsonData.slice(1) as any[][];
      
      console.log("Headers:", headers);
      console.log("Data rows count:", dataRows.length);
      
      // Validate headers
      if (!headers || headers.length === 0) {
        toast({
          title: "Error",
          description: "Header tidak ditemukan dalam file",
          variant: "destructive",
        });
        return;
      }
      
      // Convert rows to objects
      const processedData = dataRows
        .filter(row => {
          // Filter out completely empty rows
          return row && row.some(cell => cell !== null && cell !== undefined && String(cell).trim() !== "");
        })
        .map((row, index) => {
          const obj: any = {};
          headers.forEach((header, headerIndex) => {
            if (header && header.trim()) {
              const cellValue = row[headerIndex];
              // Convert cell value to string and trim
              obj[header.trim()] = cellValue !== null && cellValue !== undefined ? String(cellValue).trim() : "";
            }
          });
          return obj;
        });

      console.log("Processed data:", processedData);

      if (processedData.length === 0) {
        toast({
          title: "Error",
          description: "Tidak ada data valid ditemukan dalam file",
          variant: "destructive",
        });
        return;
      }

      // Validate that we have some required data
      const hasValidData = processedData.some(row => {
        return Object.values(row).some(value => value && String(value).trim() !== "");
      });

      if (!hasValidData) {
        toast({
          title: "Error",
          description: "Tidak ada data valid ditemukan dalam file",
          variant: "destructive",
        });
        return;
      }

      console.log("Calling onImport with processed data");
      await onImport(processedData);
      
      // Reset form on success
      setIsOpen(false);
      setFile(null);
      
    } catch (error) {
      console.error("Import error:", error);
      toast({
        title: "Error",
        description: `Gagal mengimport file: ${error instanceof Error ? error.message : 'Unknown error'}`,
        variant: "destructive",
      });
    }
  };

  return (
    <Dialog open={isOpen} onOpenChange={(open) => {
      setIsOpen(open);
      if (!open) {
        setFile(null);
      }
    }}>
      <DialogTrigger asChild>
        <Button variant="outline" className="border-teal-600 text-teal-600 hover:bg-teal-50">
          <Upload className="h-4 w-4 mr-2" />
          Import {title}
        </Button>
      </DialogTrigger>
      <DialogContent className="max-w-md">
        <DialogHeader>
          <DialogTitle>Import {title}</DialogTitle>
        </DialogHeader>
        <div className="space-y-4">
          <div className="space-y-2">
            <Label>1. Download Template</Label>
            <Button
              type="button"
              variant="outline"
              onClick={downloadTemplate}
              className="w-full"
            >
              <Download className="h-4 w-4 mr-2" />
              Download Excel Template
            </Button>
            <p className="text-xs text-gray-600">
              Download template, isi dengan data Anda, lalu upload kembali.
            </p>
          </div>

          <div className="space-y-2">
            <Label htmlFor="file">2. Upload File</Label>
            <Input
              id="file"
              type="file"
              accept=".xlsx,.xls,.csv"
              onChange={handleFileChange}
              key={file ? file.name : 'empty'} // Force re-render when file changes
            />
            {file && (
              <div className="flex items-center space-x-2 text-sm text-green-600 bg-green-50 p-2 rounded">
                <FileText className="h-4 w-4" />
                <span>{file.name} ({(file.size / 1024).toFixed(1)} KB)</span>
              </div>
            )}
            <p className="text-xs text-gray-600">
              Format yang didukung: .xlsx, .xls, .csv (maksimal 5MB)
            </p>
          </div>

          <div className="flex justify-end space-x-2">
            <Button
              type="button"
              variant="outline"
              onClick={() => {
                setIsOpen(false);
                setFile(null);
              }}
            >
              Batal
            </Button>
            <Button
              onClick={handleImport}
              disabled={!file || isLoading}
              className="bg-teal-600 hover:bg-teal-700"
            >
              {isLoading ? "Mengimport..." : "Import"}
            </Button>
          </div>
        </div>
      </DialogContent>
    </Dialog>
  );
}
