import { api } from "encore.dev/api";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

interface Siswa {
  id: string;
  nisn: string;
  nis: string;
  nama_lengkap: string;
  tempat_lahir: string;
  tanggal_lahir: string;
  jenis_kelamin: string;
  agama: string;
  alamat: string;
  no_telepon: string;
  nama_ayah: string;
  nama_ibu: string;
  pekerjaan_ayah: string;
  pekerjaan_ibu: string;
  kelas: string;
  tahun_masuk: number;
  status: string;
}

interface ListSiswaResponse {
  siswa: Siswa[];
  total: number;
}

interface ListSiswaParams {
  madrasah_id?: string;
  search?: string;
  kelas?: string;
  status?: string;
  limit?: number;
  offset?: number;
}

// Lists siswa by madrasah
export const listSiswa = api<ListSiswaParams, ListSiswaResponse>(
  { expose: true, method: "GET", path: "/siswa" },
  async (params) => {
    const { madrasah_id, search = "", kelas, status, limit = 50, offset = 0 } = params;
    
    let whereClause = "";
    let searchParams: any[] = [];
    let paramIndex = 1;
    
    if (madrasah_id) {
      whereClause = `WHERE madrasah_id = $${paramIndex}`;
      searchParams.push(madrasah_id);
      paramIndex++;
    }
    
    if (search) {
      const searchCondition = `(nama_lengkap ILIKE $${paramIndex} OR nisn ILIKE $${paramIndex} OR nis ILIKE $${paramIndex})`;
      whereClause = whereClause ? `${whereClause} AND ${searchCondition}` : `WHERE ${searchCondition}`;
      searchParams.push(`%${search}%`);
      paramIndex++;
    }
    
    if (kelas) {
      const kelasCondition = `kelas = $${paramIndex}`;
      whereClause = whereClause ? `${whereClause} AND ${kelasCondition}` : `WHERE ${kelasCondition}`;
      searchParams.push(kelas);
      paramIndex++;
    }
    
    if (status) {
      const statusCondition = `status = $${paramIndex}`;
      whereClause = whereClause ? `${whereClause} AND ${statusCondition}` : `WHERE ${statusCondition}`;
      searchParams.push(status);
      paramIndex++;
    }
    
    const siswaList = await db.rawQueryAll<Siswa>(
      `SELECT id, nisn, nis, nama_lengkap, tempat_lahir, tanggal_lahir, jenis_kelamin, agama, alamat, 
              no_telepon, nama_ayah, nama_ibu, pekerjaan_ayah, pekerjaan_ibu, kelas, tahun_masuk, status
       FROM siswa ${whereClause} 
       ORDER BY nama_lengkap 
       LIMIT $${paramIndex} OFFSET $${paramIndex + 1}`,
      ...searchParams, limit, offset
    );
    
    const totalResult = await db.rawQueryRow<{ count: number }>(
      `SELECT COUNT(*) as count FROM siswa ${whereClause}`,
      ...searchParams
    );
    
    return {
      siswa: siswaList,
      total: totalResult?.count || 0
    };
  }
);
