import { api, APIError } from "encore.dev/api";
import { authHandler } from "encore.dev/auth";
import { SQLDatabase } from "encore.dev/storage/sqldb";

const db = SQLDatabase.named("madrasah");

interface LoginRequest {
  username: string;
  password: string;
}

interface LoginResponse {
  token: string;
  userType: "superadmin" | "admin" | "operator";
  madrasahId?: string;
  madrasahName?: string;
}

interface AuthData {
  userID: string;
  userType: "superadmin" | "admin" | "operator";
  madrasahId?: string;
}

// Login endpoint for madrasah and superadmin
export const login = api<LoginRequest, LoginResponse>(
  { expose: true, method: "POST", path: "/auth/login" },
  async (req) => {
    const { username, password } = req;

    // Check for superadmin login
    if (username === "superadmin" && password === "admin123") {
      const token = generateToken("superadmin", "superadmin");
      return {
        token,
        userType: "superadmin"
      };
    }

    // Check for admin login (NSM as username, NPSN as password)
    const madrasahAdmin = await db.queryRow<{
      id: string;
      nsm: string;
      npsn: string;
      nama_madrasah: string;
    }>`
      SELECT id, nsm, npsn, nama_madrasah 
      FROM madrasah 
      WHERE nsm = ${username} AND npsn = ${password}
    `;

    if (madrasahAdmin) {
      const token = generateToken(madrasahAdmin.id, "admin");
      return {
        token,
        userType: "admin",
        madrasahId: madrasahAdmin.id,
        madrasahName: madrasahAdmin.nama_madrasah
      };
    }

    // Check for operator login (NSM-OP as username, custom password)
    const operatorMatch = username.match(/^(.+)-OP$/i);
    if (operatorMatch) {
      const nsm = operatorMatch[1];
      const madrasahOperator = await db.queryRow<{
        id: string;
        nsm: string;
        nama_madrasah: string;
        operator_registered: boolean;
        operator_password: string;
      }>`
        SELECT id, nsm, nama_madrasah, operator_registered, operator_password
        FROM madrasah 
        WHERE nsm = ${nsm}
      `;

      if (madrasahOperator) {
        if (!madrasahOperator.operator_registered) {
          throw APIError.permissionDenied("Operator belum terdaftar. Silakan registrasi terlebih dahulu.");
        }

        const hashedPassword = Buffer.from(password).toString('base64');
        if (madrasahOperator.operator_password !== hashedPassword) {
          throw APIError.unauthenticated("Password salah");
        }

        const token = generateToken(madrasahOperator.id, "operator");
        return {
          token,
          userType: "operator",
          madrasahId: madrasahOperator.id,
          madrasahName: madrasahOperator.nama_madrasah
        };
      }
    }

    throw APIError.unauthenticated("Invalid credentials");
  }
);

// Simple token generation (in production, use proper JWT)
function generateToken(userId: string, userType: string): string {
  const payload = {
    userId,
    userType,
    timestamp: Date.now()
  };
  return Buffer.from(JSON.stringify(payload)).toString('base64');
}

// Auth handler for protected endpoints
export const auth = authHandler<{}, AuthData>(
  async (data) => {
    // For this simple implementation, we'll extract auth from headers manually in each endpoint
    throw APIError.unauthenticated("Authentication required");
  }
);
